program lunar(input,output);
{19 Sep 2013 Corrected Ascent Stage delta-V and margin
 20 Jul 2014 Used Block 1C launcher
 11 Jun 2015 Block II launcher with updated Orion mass
 10 Jul 2015 Increase dv margin to 1.1%. Decreased SLA mass.
 26 Jun 2017 Set thrust offset angle for CPM ascent engine to zero. Increased
             boiloff rate to 0.17%.}

const m_leo  =   142195.; {kg, LEO mass after second stage separation}
      dv_eoi =      49.0; {m/s, Earth orbit insertion}
      dv_tli =    3184.9; {m/s, trans Lunar injection}
      ve_cps =    4535.6; {m/s, RL-10C-2 performance}
      ve_crs =    3432.3; {m/s, CPS RCS performance}
      ve_ds  =    2991.0; {m/s, LM desent stage engine performance}
      ve_as  =    3040.1; {m/s, LM ascent stage engine performance}
      ve_lmr =    2942.0; {m/s, LM RCS performance}
      ve_or  =    2650.0; {m/s, Orion RCS performance}
      ve_mpcv =   3069.5; {m/s, SM OMS-E}
      ms_mpcv =   16745.; {kg, MPCV CM 9887 kg, ESM 6858 kg}
      ma_mpcv =    1795.; {kg, MPCV adaptor, MPCV 510 kg, 1285 kg SLA}
      fa_lm = 1/(16.2+1); {LM adaptor/(LM + adapter mass)}
      m_crew =      125.; {kg, single crew mass}
      m_samp =      100.; {kg, Lunar samples}
      m_cab  =     2207.; {kg, LM crew cabin}
      fm_ds  =      0.07; {descent stage/LM}
      me_cps =    4*301.; {kg, RL-10-B2}
      n_mpcv =         3; {MPCV crew}
      n_lm   =         2; {LM crew}
      f_tli  =     1.011; {TLI dv margin increase}
      f_cps  =      1.01; {CPS dv margin increase}
      f_lm   =      1.01; {MPCV dv margin increase}
      f_mpcv =      1.01; {MPCV dv margin increase}
      rm_cps =      5.88; {CPS oxidiser to fuel ratio}
      f_u    =  0.005279; {MPCV and LM unusable propellant fraction}
      dv_tad =       0.6; {m/s, Transposition and docking RCS}
      dv_tcm1 =      3.8; {m/s, Trans Lunar TCM, Apollo 16}
      dv_loi =     960.4; {m/s, Lunar orbit insertion, Apollo 14}

      f_pd =        0.75; {Proportion of CPS powered descent}
      dv_pdi =      24.9; {m/s, Powered descent initiation, Apollo 16}
      dv_pdr =       5.5; {m/s, Powered descent 1 RCS}
      dv_pd  =    2041.6; {m/s, Total powered descent, Apollo 17}
      dv_dsr =       5.5; {m/s, Powered descent 2 RCS}
      dv_asr =       5.5; {m/s, Ascent RCS}
      dv_as  =    1890.0; {m/s, LM ascent, Apollo 11}

      dv_llo =       5.5; {m/s, Low Lunar orbit 1 RCS}
      dv_pc  =      46.2; {m/s, Plane change}
      dv_tei =    1168.7; {m/s, Trans Earth injection, Apollo 14}
      dv_tcm2 =      1.7; {m/s, Trans Earth TCM, Apollo 15}

      alpha_cps = 0.46718; {CPS scale factor, Boeing = 0.21539, ULA = 0.16349}
      alpha_as =  0.9707; {LM AS scale factor}
      eoa_as   =       0; {degrees, AS engine offset angle}
      beta     =   0.848; {power factor}
      epsilon  =    1e-6; {a small number}
      t_leo    =    0.25; {days, time in LEO}
      t_tl     =    3.50; {days, time in TLI transit}
      t_llo    =    1.25; {days, time in LLO}
      f_bo     =  0.0017; {fraction boiloff per day}
      f_ul     =    1.07; {ullage volume increase}
      d_o      =   1149.; {kg/kL, oxidiser density}
      d_f      =    70.9; {kg/kL, fuel density}
      D_cps    =  8.4074; {m, tank diameter}
      d_lmo    =   1431.; {kg/kL, N2O4 density}
      d_lmf    =   881.8; {kg/kL, Aerozine-50 density}
      rm_lm    =     1.6; {LM mixture ratio}

      S4_N      =      4; {no. of engines}
      S4_F      = 110093; {N,  vacuum thrust}
      S4_V      = 4535.6; {m/s, vacuum exhaust speed, full nozzle}
      S4_MR     =   5.88; {oxidiser to fuel mixture ratio}
      S4_Ro_r   = 1053.9; {kg/s, S-II reference oxidiser rate}
      S4_Rf_r   =  190.4; {kg/s, S-II reference fuel rate}
      S4_Mito_r =    679; {kg, reference LOX In Tank at Separation}
      S4_Mbto_r =    787; {kg, reference LOX Below Tank at Separation}
      S4_Mugo_r =   2254; {kg, reference LOX ullage gas}
      S4_Mpo_r  = 379876; {kg, total LOX at liftoff} 
      S4_Mitf_r =   1505; {kg, reference Fuel In Tank at Separation}
      S4_Mbtf_r =    123; {kg, reference Fuel Below Tank at Separation}
      S4_Mugf_r =    599; {kg, reference fuel ullage gas}
      S4_Mpf_r  =  72476; {kg, total Fuel at liftoff} 
      S4_MR_r   =    4.8; {reference oxidiser to fuel mixture ratio}
      S4_Rf = S4_N*S4_F/(S4_V*(1+S4_MR));
                            {kg/s, fuel mass rate}
      mp_bto = round(S4_Mbto_r*S4_Rf*S4_MR/S4_Ro_r);
                            {kg, below tank oxidiser}
      mp_btf = round(S4_Mbtf_r*S4_Rf/S4_Rf_r);
                            {kg, below tank fuel}
      mp_bt = mp_bto+mp_btf;
                            {kg, below tank propellant}
      S4_Mfb_r  = S4_Mitf_r-S4_Mito_r/S4_MR_r;
                            {kg, reference fuel bias}
      mp_fb = round(S4_Mfb_r*S4_Rf/S4_Rf_r); 
                            {kg, fuel bias propellant}
      f_ugo = S4_Mugo_r/(S4_Mpo_r-S4_Mbto_r-S4_Mugo_r);
      f_ugf = S4_Mugf_r/(S4_Mpf_r-S4_Mbtf_r-S4_Mugf_r);

var mf,mp_mpcv,m_mpcv,mi,mp_cps,ms_cps,m_cps,mpo,m_lm:double;
    ma_lm,mc,mco,ms_ds,ms_as,mp_lm,ms_lm,mp_ug,mp_lm_u:double;
    mp_eoi,mp_tad,mp_tli,mp_tcm1,mp_loi,mp_pdi,mp_pdr,mp_pd,mp_dsr:double;
    mp_ds,mp_asr,mp_as,mp_llo,mp_pc,mp_tei,mp_tcm2:double;
    mb_leo,mb_tl,mb_llo,mp_cps_r,mp_lm_r,mp_mpcv_r,mp_mpcv_u:double;
    mp_ugo,mp_ugf,mp_u,V_o,V_f,V_cps,L_cps,G_cps,V_lmo,V_lmf,D_lm:double;

function mpf(mf,dv,ve,f:double):double;
begin{mpf}
  mpf := mf*(exp(f*dv/ve)-1);
end;{mpf}

function mpi(mi,dv,ve,f:double):double;
begin{mpi}
  mpi := mi*(1-exp(-f*dv/ve));
end;{mpi}

function shell(D,H,V:double):double;
{Uses Newton's method to solve shell equation}

const epsilon = 1e-6;
var G,Gold:double;

function f(G:double):double;
begin{f}
  f := 0.5*pi*sqr(D)*(2*H/3 + G*(sqr(G/H)/3-1));
end;{f}

function fp(G:double):double;
begin{fp}
  fp := 0.5*pi*sqr(D)*(sqr(G/H)-1);
end;{fp}

begin{shell}
  G := 0;
  repeat
    Gold := G;
    G := Gold -(f(Gold)-V)/fp(Gold);
  until abs(G-Gold) < epsilon;
  shell := G;
end;{shell}

begin{lunar}
  mp_mpcv_u := 0;
  repeat
    mpo := mp_mpcv_u;
    mf := ms_mpcv + n_mpcv*m_crew + m_samp + mp_mpcv_u;
    mp_tcm2 := mpf(mf,dv_tcm2,ve_or,f_mpcv);

    mf := mf+mp_tcm2;
    mp_tei := mpf(mf,dv_tei,ve_mpcv,f_mpcv);

    mf := mf+mp_tei;
    mp_llo := mpf(mf,dv_llo,ve_or,f_mpcv);

    mf := mf+mp_llo-m_samp-n_lm*m_crew;
    mp_pc := mpf(mf,dv_pc,ve_mpcv,f_mpcv);

    mf := mf+mp_pc+n_lm*m_crew;
    mp_tad := mpf(mf,dv_tad,ve_or,f_mpcv);

    m_mpcv := mf+mp_tad;
    mp_mpcv := (mp_tad + mp_pc + mp_llo + mp_tei + mp_tcm2)/(1-f_u);
    mp_mpcv_u := f_u*mp_mpcv;
  until abs(mp_mpcv_u-mpo) < epsilon;

  mp_eoi := mpi(m_leo,dv_eoi,ve_cps,f_cps);
  mp_cps := mp_eoi;
  repeat
    mpo := mp_cps;

    mb_leo := mp_cps*t_leo*f_bo;
    mi := m_leo-mp_eoi-mp_tad-mb_leo-ma_mpcv;
    mp_tli := mpi(mi,dv_tli,ve_cps,f_tli);

    mi := mi-mp_tli;
    mp_tcm1 := mpi(mi,dv_tcm1,ve_crs,f_cps);
    mb_tl := mp_cps*t_tl*f_bo;

    mi := mi-mp_tcm1-mb_tl;
    mp_loi := mpi(mi,dv_loi,ve_cps,f_cps);
    mb_llo := mp_cps*t_llo*f_bo;

    mi := mi-mp_loi-mb_llo-m_mpcv+mp_tad+n_lm*m_crew;
    mp_pdi := mpi(mi,dv_pdi,ve_crs,f_cps);

    mi := mi-mp_pdi;
    mp_pdr := mpi(mi,dv_pdr,ve_crs,f_cps);
    mp_pd := mpi(mi,dv_pd*f_pd,ve_cps,f_cps);

    mp_cps := mp_eoi + mb_leo + mp_tli + mp_tcm1 + mb_tl + mp_loi + mb_llo + 
              mp_pdi + mp_pdr + mp_pd;
    mp_ugf := round(f_ugf*(mp_cps/(1+rm_cps)+mp_fb));
    mp_ugo := round(f_ugo*mp_cps/(1+1/rm_cps));
    mp_ug := mp_ugf + mp_ugo;
    mp_cps := mp_cps + mp_ug + mp_fb + mp_bt;
  until abs(mp_cps-mpo) < epsilon;
  ms_cps := me_cps + alpha_cps*exp(beta*ln(mp_cps));
  m_cps := mp_cps + ms_cps;

  mi := mi-mp_pdr-mp_pd-mp_ug-mp_fb-mp_bt-ms_cps;
  ma_lm := (mi-n_lm*m_crew)*fa_lm;

  mi := mi-ma_lm;
  mp_dsr := mpi(mi,dv_dsr,ve_lmr,f_lm);
  mp_ds := mpi(mi,dv_pd*(1-f_pd),ve_ds,f_lm);

  mi := mi-mp_dsr-mp_ds;
  ms_ds := mi*fm_ds;

  m_lm := mi;
  mc := 0;
  repeat
    mi := m_lm-mc+m_samp-ms_ds;
    mp_asr := mpi(mi,dv_asr,ve_lmr,f_lm);
    mp_as := mpi(mi,dv_as,ve_as*cos(eoa_as*pi/180),f_lm);

    mp_lm := (mp_dsr + mp_ds + mp_asr + mp_as)/(1-f_u);
    mp_lm_u := f_u*mp_lm;
    ms_as := m_cab + alpha_as*exp(beta*ln(mp_lm));
    ms_lm := ms_ds + ms_as;
    mco := mc;
    mc := m_lm - n_lm*m_crew - ms_lm - mp_asr - mp_as - mp_lm_u;
  until abs(mc-mco) < epsilon;
  m_lm := m_lm + mp_dsr + mp_ds + ma_lm - n_lm*m_crew;

  writeln('CSM: mp_tad = ',mp_tad:5:0,' kg, mp_pc = ',mp_pc:5:0,
          ' kg, tmp_llo = ',mp_llo:5:0,' kg, mp_tei = ',mp_tei:5:0,' kg');
  writeln('     mp_tcm2 = ',mp_tcm2:5:0,' kg, mp_mpcv_u = ',mp_mpcv_u:5:0,
          ' kg, mp_mpcv = ',mp_mpcv:5:0,' kg');
  writeln('     ms_mpcv = ',ms_mpcv:5:0,' kg, m_crew = ',m_crew*n_mpcv:5:0,
          ' kg, m_mpcv = ',m_mpcv:5:0,' kg');
  writeln('CPS: mp_eoi = ',mp_eoi:5:0,' kg, mb_leo = ',mb_leo:5:0,
          ' kg, mp_tli = ',mp_tli:5:0,' kg, mp_tcm = ',mp_tcm1:5:0,' kg');
  writeln('     mb_tl = ',mb_tl:5:0,' kg, mp_loi = ',mp_loi:5:0,
          ' kg, mb_llo = ',mb_llo:5:0,' kg, tmp_pdi = ',mp_pdi:5:0,' kg');
  writeln('     mp_pd = ',mp_pd:5:0,' kg, mp_pdr = ',mp_pdr:5:0,
          ' kg, mp_ug = ',mp_ug:5:0,' kg, mp_bt = ',mp_bt:5,' kg');
  writeln('     mp_fb = ',mp_fb:5,' kg, mp_cps = ',mp_cps:5:0,
          ' kg, ms_cps = ',ms_cps:5:0,' kg, m_cps = ',m_cps:5:0,' kg');
  writeln('LM:  mp_dsr = ',mp_dsr:5:0,' kg, mp_ds = ',mp_ds:5:0,
          ' kg, mp_asr = ',mp_asr:5:0,' kg, mp_as = ',mp_as:5:0,' kg');
  writeln('     mp_lm_u = ',mp_lm_u:5:0,' kg, mp_lm = ',mp_lm:5:0,
          ' kg, ms_ds = ',ms_ds:5:0,' kg, ms_as = ',ms_as:5:0,' kg');
  writeln('     ms_lm = ',ms_lm:5:0,' kg, ma_lm = ',ma_lm:5:0,
          ' kg, m_crew = ',n_lm*m_crew:5:0,' kg, m_samp = ',m_samp:5:0,' kg');
  writeln('     mc = ',mc:5:0,' kg, m_lm = ',m_lm:5:0,' kg');
  mi := m_cps+m_lm+ma_mpcv+m_mpcv;
  write('All = ',mi:5:0,' kg, ');
  mi := mi-mp_eoi;
  write('LEO = ',mi:5:0,' kg, ');
  mi := mi-ma_mpcv-mp_tad-mb_leo-mp_tli;
  writeln('TLI = ',mi:5:0,' kg');
  writeln('dv_pd = ',dv_pd:6:1,' m/s, dv_ds = ',dv_pd*(1-f_pd):5:1,
          ' m/s, dv_as = ',dv_as:5:1,' m/s');

  mi := m_mpcv;
  mp_tad := mpi(mi,dv_tad,ve_or,1);

  mi := mi-mp_tad-n_lm*m_crew;
  mp_pc := mpi(mi,dv_pc,ve_mpcv,1);

  mi := mi-mp_pc+m_samp+n_lm*m_crew;
  mp_llo := mpi(mi,dv_llo,ve_or,1);

  mi := mi-mp_llo;
  mp_tei := mpi(mi,dv_tei,ve_mpcv,1);

  mi := mi-mp_tei;
  mp_tcm2 := mpi(mi,dv_tcm2,ve_or,1);

  mp_mpcv_r := mp_mpcv - (mp_tad + mp_pc + mp_llo + mp_tei + mp_tcm2 + 
               mp_mpcv_u);

  mi := m_leo;
  mp_eoi := mpi(mi,dv_eoi,ve_cps,1);
  mi := mi-mp_eoi-mp_tad-mb_leo-ma_mpcv;
  mp_tli := mpi(mi,dv_tli,ve_cps,1);

  mi := mi-mp_tli;
  mp_tcm1 := mpi(mi,dv_tcm1,ve_crs,1);

  mi := mi-mp_tcm1-mb_tl;
  mp_loi := mpi(mi,dv_loi,ve_cps,1);

  mi := mi-mp_loi-mb_llo-m_mpcv+mp_tad+n_lm*m_crew;
  mp_pdi := mpi(mi,dv_pdi,ve_crs,1);

  mi := mi-mp_pdi;
  mp_pdr := mpi(mi,dv_pdr,ve_crs,1);
  mp_pd := mpi(mi,dv_pd*f_pd,ve_cps,1);

  mp_cps_r := mp_cps - (mp_eoi + mb_leo + mp_tli + mp_tcm1 + mb_tl + mp_loi + 
              mb_llo + mp_pdi + mp_pdr + mp_pd + mp_ug + mp_bt + mp_fb);

  mi := mi-mp_pdr-mp_pd-mp_cps_r-mp_ug-mp_bt-mp_fb-ms_cps-ma_lm;
  mp_dsr := mpi(mi,dv_dsr,ve_lmr,1);
  mp_ds := mpi(mi,dv_pd*(1-f_pd),ve_ds,1);

  mi := mi-mp_dsr-mp_ds-mc+m_samp-ms_ds;
  mp_asr := mpi(mi,dv_asr,ve_lmr,1);
  mp_as := mpi(mi,dv_as,ve_as*cos(eoa_as*pi/180),1);

  mp_lm_r := mp_lm - (mp_dsr + mp_ds + mp_asr + mp_as + mp_lm_u);

  writeln;
  writeln('CSM: mp_tad    = ',mp_tad:6:0,' kg');
  writeln('     mp_pc     = ',mp_pc:6:0,' kg');
  writeln('     tmp_llo   = ',mp_llo:6:0,' kg');
  writeln('     mp_tei    = ',mp_tei:6:0,' kg');
  writeln('     mp_tcm2   = ',mp_tcm2:6:0,' kg');
  writeln('     mp_mpcv_r = ',mp_mpcv_r:6:0,' kg');
  writeln('     mp_mpcv_u = ',mp_mpcv_u:6:0,' kg');
  writeln('     mp_mpcv   = ',mp_mpcv:6:0,' kg');
  writeln('     ms_mpcv   = ',ms_mpcv:6:0,' kg');
  writeln('     m_crew    = ',m_crew*n_mpcv:6:0,' kg');
  writeln('     m_mpcv    = ',m_mpcv:6:0,' kg');
  writeln;
  writeln('CPS: mp_eoi    = ',mp_eoi:6:0,' kg');
  writeln('     mb_leo    = ',mb_leo:6:0,' kg');
  writeln('     mp_tli    = ',mp_tli:6:0,' kg');
  writeln('     mp_tcm    = ',mp_tcm1:6:0,' kg');
  writeln('     mb_tl     = ',mb_tl:6:0,' kg');
  writeln('     mp_loi    = ',mp_loi:6:0,' kg');
  writeln('     mb_llo    = ',mb_llo:6:0,' kg');
  writeln('     tmp_pdi   = ',mp_pdi:6:0,' kg');
  writeln('     mp_pd     = ',mp_pd:6:0,' kg');
  writeln('     mp_pdr    = ',mp_pdr:6:0,' kg');
  writeln('     mp_cps_r  = ',mp_cps_r:6:0,' kg');
  writeln('     mp_ug     = ',mp_ug:6:0,' kg');
  writeln('     mp_bt     = ',mp_bt:6,' kg');
  writeln('     mp_fb     = ',mp_fb:6,' kg');
  writeln('     mp_cps    = ',mp_cps:6:0,' kg');
  writeln('     ms_cps    = ',ms_cps:6:0,' kg');
  writeln('     m_cps     = ',m_cps:6:0,' kg');
  writeln;
  writeln('LM:  mp_dsr    = ',mp_dsr:6:0,' kg');
  writeln('     mp_ds     = ',mp_ds:6:0,' kg');
  writeln('     mp_asr    = ',mp_asr:6:0,' kg');
  writeln('     mp_as     = ',mp_as:6:0,' kg');
  writeln('     mp_lm_r   = ',mp_lm_r:6:0,' kg');
  writeln('     mp_lm_u   = ',mp_lm_u:6:0,' kg');
  writeln('     mp_lm     = ',mp_lm:6:0,' kg');
  writeln('     ms_lcm    = ',ms_ds:6:0,' kg');
  writeln('     ms_cpm    = ',ms_as:6:0,' kg');
  writeln('     ms_lm     = ',ms_lm:6:0,' kg');
  writeln('     ma_lm     = ',ma_lm:6:0,' kg');
  writeln('     m_crew    = ',n_lm*m_crew:6:0,' kg');
  writeln('     m_samp    = ',m_samp:6:0,' kg');
  writeln('     mc        = ',mc:6:0,' kg');
  writeln('     m_lm      = ',m_lm:6:0,' kg');
  writeln;
  mi := m_cps+m_lm+ma_mpcv+m_mpcv;
  writeln('All    = ',mi:7:0,' kg');
  mi := mi-mp_eoi;
  writeln('LEO    = ',mi:7:0,' kg');
  mi := mi-ma_mpcv-mp_tad-mb_leo-mp_tli;
  writeln('TLI    = ',mi:7:0,' kg');
  writeln('dv_pd  = ',dv_pd:7:1,' m/s');
  writeln('dv_ds  = ',dv_pd*(1-f_pd):7:1,' m/s');
  writeln('dv_as  = ',dv_as:7:1,' m/s');

  mp_u := mp_cps - mp_fb - mp_bt - mp_ug;
  V_o := f_ul*(mp_u/(1+1/rm_cps)+mp_ugo)/d_o;
  V_f := f_ul*(mp_u/(1+rm_cps)+mp_ugf+mp_fb)/d_f;
  V_cps := V_o + V_f;
  L_cps := 4*(V_cps/(pi*sqr(D_cps))-D_cps/9);
  G_cps := shell(D_cps,D_cps/3,V_o);
  writeln;
  writeln('mp_bto = ',mp_bto:7,' kg');
  writeln('mp_btf = ',mp_btf:7,' kg');
  writeln('mp_ugo = ',mp_ugo:7:0,' kg');
  writeln('mp_ugf = ',mp_ugf:7:0,' kg');
  writeln('mp_ms  = ',mp_u-mp_cps_r:7:0,' kg');
  writeln('V_cpso = ',V_o:7:3,' kL');
  writeln('V_cpsf = ',V_f:7:3,' kL');
  writeln('V_cps  = ',V_cps:7:3,' kL');
  writeln('L      = ',L_cps:7:3,' m');
  writeln('G      = ',G_cps:7:3,' m');

  V_lmo := f_ul*mp_lm/(d_lmo*(1+1/rm_lm));
  V_lmf := f_ul*mp_lm/(d_lmf*(1+rm_lm));
  D_lm := exp(ln(3*V_lmf/pi)/3);
  writeln;
  writeln('V_lmo  = ',V_lmo:7:3,' kL');
  writeln('V_lmf  = ',V_lmf:7:3,' kL');
  writeln('D_lm   = ',D_lm:7:3,' m');
end.{lunar}