{rocket.pas written by Steven S. Pietrobon 23 Oct 1996.
 Revised 7 Mar 1999, 6 Jan 2000, 17 Aug 2006,
 20 Jul 2013  Changed Fd to Ae
 21 Jul 2013  Fixed error when inverse term in arctan is zero.
              Limited thrust to zero when less than zero. Used default pi.
 28 Jul 2013  Fixed calculation of alpha.
 15 Sep 2013  Added determination of maximum loads with fmax.
 19 Oct 2013  Fixed fmax calculation for correct area.}

const error = 1e-6;
      Nv = 4;

type path = (air,orb,vac);

var dF,dRp,Ae,x1s,Ar,dt,dp,Me,t0,x0,x1,h0,h1,r0,p0,m0,m1,f0,a,Te,ho,dv:double;
    vi,hp,ha,e,sma,maxa,alpha,pow,amax,hmax,tt:double;
    traj:path;
    iorb,fair,simout:boolean;
    question:text;
    speed,height,accel,dynpres:text;
    continue,sqrc,degc:char;
    output_name:string;
    x0d,x1d,h0d,h1d,r0d,p0d,m0d,m1d,f0d,dvd:array[1..Nv] of double;
    fmax,tfmax,mfmax,Afmax,force,a_max,ta_max,Pqmax,tPqmax,alo,flo,mlo:double;

procedure init;
{Initialisation procedure. Steven S. Pietrobon, 24 Jul 1995}
var I:integer;
begin{init}
  a_max := 0;
  ta_max := 0;
  Pqmax := 0;
  tPqmax := 0;

  for I := 1 to Nv do
    begin{init arrays}
      x0d[I] := 0;
      x1d[I] := 0;
      h0d[I] := 0;
      h1d[I] := 0;
      r0d[I] := 0;
      p0d[I] := 0;
      m0d[I] := 0;
      m1d[I] := 0;
      f0d[I] := 0
   end;{init arrays}
  write('Enter output filename (return is standard output): ');
  readln(output_name);
  assign(output,output_name);
  rewrite(output);
  assign(question,'');
  rewrite(question);
  simout := output_name <> '';
  if simout
    then begin{file out}
           degc := char(176);
           sqrc := char(178);
         end{file out}
    else begin{dosbox}
           degc := char(248);
           sqrc := char(253);
         end;{dosbox}

  if simout then
    begin{simout}
      assign(speed,'speed.dat');
      rewrite(speed);
      assign(height,'height.dat');
      rewrite(height);
      assign(accel,'accel.dat');
      rewrite(accel);
      assign(dynpres,'dynpres.dat');
      rewrite(dynpres);
    end;{simout}

  init_atmosphere;
  writeln;
  write('   t      a     vi     h0       r0      alpha  beta  theta     Pq');
  writeln('     m0+Me');
  write('  sec    m/s',sqrc,'  m/s   metres   metres     deg    deg   deg      Pa');
  writeln('       kg');
  write('-------------------------------------------------------------------');
  write('----------');
  writeln;
end;{init}

procedure close_sim;
begin{close}
  close(output);
  close(question);
  if simout then
    begin{simout}
      close(speed);
      close(height);
      close(accel);
      close(dynpres);
    end;{simout}
end;{close}

procedure writeqn(str:string;
                  var x:double);
{Writes a question and reads an answer}
begin{write question}
  write(question,str);
  readln(x);
  if simout then
    writeln(str,x:6:3);
end;{write question}

procedure trajectory;
{Calculates the incremental trajectory of a rocket.
 Steven S. Pietrobon 24 Jul 1995, 20 Jul 2013
 traj = air: Use for all liftoffs (even in a vacuum).
             Rocket follows air velocity vector.
 traj = orb: Angle of attack automatically adjusted to bring
             rocket into orbit. Vacuum or air operation.
 traj = vac: Vacuum only. Rocket follows velocity vector.
 Inputs:    dF  (N/s, change in vacuum thrust)
            Ae  (m^2, engine nozzle exit area)
            dRp (kg/s^2, change in propellant mass rate)
            Ar  (m^2, cross sectional area of rocket)
            x1s (m/s, surface air speed at launch site)
            Re  (m, equatorial radius of planet)
            mu  (m^3/s^2, G*M, gravitational parameter)
            dt  (s, time increment)
            dp  (rad, angle of attack increment)
            Me  (kg, empty mass)
            ho  (1/s^2, orbital reach factor)
            pi  (rad, 180 degrees)
            hmin (m, perigee altitude)
            maxa (maximum angle of attack for orb)
            traj (trajectory type)
            iorb (initialise orbit)
            fair (follow air vector)
            alpha (rad, angle of attack)
            pow (orbital law factor)
 Outputs:   a   (m/s^2, acceleration strain)
            Pq  (Pa, dynamic pressure)
            sma (m, semi-major axis of orbit)
            e   (eccentricty of orbit)
            hp  (m, perigee altitude of orbit)
            ha  (m, apogee altitude of orbit)
 States:    t0  (s, time)
            x0  (m, distance)
            x1  (m/s, distance speed)
            h0  (m, altitude)
            h1  (m/s, altitude speed)
            r0  (m, range)
            p0  (rad, angle of attack offset)
            m0  (kg, propellant mass) 
            m1  (kg/s, propellant rate)
            f0  (N, vacuum thrust) 
            dv  (m/s, vacuum and zero gravity delta-V)}

var x1t,h0t,h1t,p0t,m0t,m1t,f0t:double;
    r,x1a,va,theta,temp,thrust,drag,mass,Pq,angle,beta,fp,Ph,Dh,vs:double;
    I,J:integer;

function cd(M:double):double;
{Coefficient of drag from "The Mars Project" by von Braun.
 Inputs:    M   (Mach number)
 Outputs:   cd  (coefficient of drag)}
begin{cd}
  if M < 5
  then case round(M-0.5) of
         0: cd := 0.4;
         1: if M < 1.4
              then cd := 0.8
              else cd := 0.8 - 0.11*(M-1.4)/0.6;
         2: cd := 0.69 - 0.1*(M-2);
         3,4: cd := 0.59 - 0.02*(M-3)
       end{case}
  else cd := 0.55
end;{cd}

begin{trajectory}
  for I := 1 to 4 do
    begin{compute data}
      if I = 1 
        then begin{initialise}
                x1t := x1;
                h0t := h0;
                h1t := h1;
                p0t := p0;
                m0t := m0;
                m1t := m1;
                f0t := f0;
              end{initialise}
         else begin{iterate}
                if I = 4
                  then temp := 1
                  else temp := 0.5;
                J := I-1;
                x1t := x1 + x1d[J]*temp;
                h0t := h0 + h0d[J]*temp;
                h1t := h1 + h1d[J]*temp;
                p0t := p0 + p0d[J]*temp;
                m0t := m0 + m0d[J]*temp;
                m1t := m1 + m1d[J]*temp;
                f0t := f0 + f0d[J]*temp;
              end;{iterate}

      r := Re+h0t;
      mass := Me + m0t;
      x0d[I] := x1t*dt;
      h0d[I] := h1t*dt;
      r0d[I] := (x1t*Re/r - x1s)*dt;
      p0d[I] := dp*dt;
      m0d[I] := m1t*dt;
      m1d[I] := dRp*dt;
      f0d[I] := dF*dt;
      dvd[I] := f0t*dt/mass;

      beta := arctan(h1t/x1t);
      if (traj = air) or (traj = orb) then
        begin{air drag}
          x1a := x1t - x1s*r/Re;
          va := sqrt(x1a*x1a + h1t*h1t);
          if x1a < error then theta := pi/2
                         else theta := arctan(h1t/x1a);
          atmosphere(h0t,Ph,Dh,vs);
          thrust := f0t - Ae*Ph;
          if thrust < 0 then
            thrust := 0;
          Pq := Dh*va*va/2;
          drag := Pq*Ar*cd(va/vs);
        end;{air drag}

      case traj of
        air: begin{air}
               if fair = true
                  then angle := theta + p0t
                  else angle := beta + alpha + p0t;
               x1d[I] := ((thrust*cos(angle)-drag*cos(theta))/mass 
                         - x1t*h1t/r)*dt;
               h1d[I] := ((thrust*sin(angle)-drag*sin(theta))/mass 
                         + x1t*x1t/r - mu/(r*r))*dt
             end;{air}
        orb: begin{orbit}
               if iorb = true then
                 begin{initialise orbital parameter}
                   temp := exp(pow*ln(abs(h1t)));
                   ho := ((thrust*sin(beta+alpha)-drag*h1t/va)/mass 
                         + x1t*x1t/r - mu/(r*r))/temp;
                   iorb := false
                 end;{initialise orbital parameter}
               temp := exp(pow*ln(abs(h1t)))*ho;
               if h1t < 0 then temp := -temp;
               fp := mass*(temp - x1t*x1t/r + mu/(r*r)) + drag*h1t/va;
               if fp+error > thrust 
                 then fp := thrust
                 else if -fp+error > thrust then fp := -thrust;
               angle := sqrt(1/sqr(fp/thrust)-1);
               if angle = 0
                 then angle := pi/2
                 else angle := arctan(1/angle);
               if fp > 0 
                 then angle := angle - beta
                 else angle := -angle - beta;
               if abs(angle) > maxa
                 then begin{max attack}
                        if angle > 0
                          then angle := maxa
                          else angle := -maxa;
                        temp := beta + angle;
                        x1d[I] := ((thrust*cos(temp)-drag*x1a/va)/mass
                                  - x1t*h1t/r)*dt;
                        h1d[I] := ((thrust*sin(temp)-drag*h1t/va)/mass
                                  + x1t*x1t/r - mu/(r*r))*dt
                      end{max attack}
                 else begin{adjust attack}
                        x1d[I] := ((sqrt(sqr(thrust)-fp*fp)-drag*x1a/va)/mass
                                  - x1t*h1t/r)*dt;
                        h1d[I] := temp*dt
                      end{adjust attack}
             end;{orbit}
        vac: begin{vacuum}
               a := f0t/mass;
               temp := beta + p0t;
               x1d[I] := (a*cos(temp) - x1t*h1t/r)*dt;
               h1d[I] := (a*sin(temp) + x1t*x1t/r - mu/(r*r))*dt
             end{vacuum}
      end{case}
    end;{compute data}

  x0 := x0 + (x0d[1] + 2*(x0d[2] + x0d[3]) + x0d[4])/6;
  x1 := x1 + (x1d[1] + 2*(x1d[2] + x1d[3]) + x1d[4])/6;
  h0 := h0 + (h0d[1] + 2*(h0d[2] + h0d[3]) + h0d[4])/6;
  h1 := h1 + (h1d[1] + 2*(h1d[2] + h1d[3]) + h1d[4])/6;
  r0 := r0 + (r0d[1] + 2*(r0d[2] + r0d[3]) + r0d[4])/6;
  p0 := p0 + (p0d[1] + 2*(p0d[2] + p0d[3]) + p0d[4])/6;
  m0 := m0 + (m0d[1] + 2*(m0d[2] + m0d[3]) + m0d[4])/6;
  m1 := m1 + (m1d[1] + 2*(m1d[2] + m1d[3]) + m1d[4])/6;
  f0 := f0 + (f0d[1] + 2*(f0d[2] + f0d[3]) + f0d[4])/6;
  dv := dv + (dvd[1] + 2*(dvd[2] + dvd[3]) + dvd[4])/6;
  t0 := t0 + dt;

  r := Re+h0;
  vi := sqrt(x1*x1+h1*h1);
  mass := Me + m0;

  beta := arctan(h1/x1);
  x1a := x1 - x1s*r/Re;
  va := sqrt(x1a*x1a + h1*h1);
  if x1a < error then theta := pi/2
                 else theta := arctan(h1/x1a);

  if (traj = air) or (traj = orb)
    then begin{air drag}
           atmosphere(h0,Ph,Dh,vs);
           Pq := Dh*va*va/2;
           thrust := f0 - Ae*Ph;
           if thrust < 0 then
             thrust := 0;
           a := thrust/mass;
           drag := Pq*cd(va/vs);
        end{air drag}
    else begin{vacuum}
           Pq := 0;
           a := f0/mass;
           drag := 0;
         end;{vacuum}
  if Me+m0 > mfmax then
    begin{check force}
      force := drag*Afmax+mfmax*a;
      if force > fmax then
        begin{new fmax}
          fmax := force;
          tfmax := t0;
        end;{new fmax}
    end;{check force}
  drag := drag*Ar;
  if simout then
    begin{find parameters}
      if a > a_max then
        begin{new a_max}
          a_max := a;
          ta_max := t0;
        end;{new a_max}
      if Pq > Pqmax then
        begin{new Pqmax}
          Pqmax := Pq;
          tPqmax := t0;
        end;{new Pqmax}
    end;{find parameters}

  case traj of
    air: if fair = true then alpha := theta-beta;
    orb: begin{orbit}
           temp := exp(pow*ln(abs(h1)))*ho;
           if h1 < 0 then temp := -temp;
           fp := mass*(temp - x1*x1/r + mu/(r*r)) + drag*h1/va;
           if fp+error > thrust
             then fp := thrust
             else if -fp+error > thrust then fp := -thrust;
           alpha := sqrt(1/sqr(fp/thrust)-1);
           if alpha = 0
             then alpha := pi/2
             else alpha := arctan(1/alpha);
           if fp > 0
             then alpha := alpha - beta
             else alpha := -alpha - beta;
           if abs(alpha) > maxa
             then begin{max attack}
                    if alpha > 0
                      then alpha := maxa
                      else alpha := -maxa
                  end{max attack}
         end;{orbit}
    vac: alpha := 0
  end;{case}

  temp := 180.0/pi;
  writeln(t0:7:2,a:6:1,round(vi):6,round(h0):8,round(r0):10,
          alpha*temp:7:2,beta*temp:7:2,theta*temp:7:2,Pq:9:1,m0+Me:10:1);
  if simout then
    begin{simout}
      writeln(speed,t0:7:2,round(vi):6);
      writeln(height,t0:7:2,h0/1000:9:3);
      writeln(accel,t0:7:2,a:8:3);
      writeln(dynpres,t0:7:2,Pq/1000:9:4);
    end;{simout}

  atmosphere(h0,Ph,Dh,vs);
  a := (f0 - Ae*Ph)/(Me+m0);
  if a < 0 then
    a := 0;
  sma := 1/(2/r - (x1*x1+h1*h1)/mu);
  temp := 1 - sqr(r*x1)/(sma*mu);
  if temp >= 0.0
    then e := sqrt(temp)
    else e := 1.0;
  hp := sma*(1-e)-Re;
  ha := sma*(1+e)-Re;
end;{trajectory}

procedure time_traj;
{Determines trajectory for a certain time Te.
 Steven S. Pietrobon 23 Dec 1994}
begin{time traj}
  while Te > dt do
    begin{increment time}
      trajectory;
      Te := Te-dt
    end;{increment time}
  dt := Te;
  trajectory
end;{time traj}

procedure traj_acc;
{Continue trajectory until amax is reached
 Steven S. Pietrobon 11 Jul 1996}
begin{traj acc}
  repeat
    trajectory;
    Te := Te-dt
  until a > amax;
  dt := -dt;
  trajectory;
  Te := Te-dt
end;{traj acc}

procedure traj_height;
{Continue trajectory until hmax is reached
 Steven S. Pietrobon 24 Jul 1995}
begin{traj height}
  repeat
    trajectory;
    Te := Te-dt
  until h0 > hmax;
  dt := -dt;
  trajectory;
  Te := Te-dt
end;{traj height}
